/*******************************************************************
 * This file is part of the Emulex Linux Device Driver for         *
 * Fibre Channel Host Bus Adapters.                                *
 * Copyright (C) 2003-2005 Emulex.  All rights reserved.           *
 * EMULEX and SLI are trademarks of Emulex.                        *
 * www.emulex.com                                                  *
 *                                                                 *
 * This program is free software; you can redistribute it and/or   *
 * modify it under the terms of version 2 of the GNU General       *
 * Public License as published by the Free Software Foundation.    *
 * This program is distributed in the hope that it will be useful. *
 * ALL EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND          *
 * WARRANTIES, INCLUDING ANY IMPLIED WARRANTY OF MERCHANTABILITY,  *
 * FITNESS FOR A PARTICULAR PURPOSE, OR NON-INFRINGEMENT, ARE      *
 * DISCLAIMED, EXCEPT TO THE EXTENT THAT SUCH DISCLAIMERS ARE HELD *
 * TO BE LEGALLY INVALID.  See the GNU General Public License for  *
 * more details, a copy of which can be found in the file COPYING  *
 * included with this package.                                     *
 *******************************************************************/

/*
 * $Id: lpfc_misc.c 2720 2005-11-28 18:58:48Z sf_support $
 */

#include <linux/version.h>
#include <linux/blkdev.h>
#include <linux/dma-mapping.h>
#include <linux/pci.h>
#include <linux/interrupt.h>
#include <linux/spinlock.h>
#include <linux/mutex.h>
#include <linux/mempool.h>
#include <linux/blkdev.h>
#include <linux/interrupt.h>
#include <linux/kthread.h>
#include <scsi/scsi.h>
#include <scsi/scsi_device.h>
#include <scsi/scsi_host.h>
#include <scsi/scsi_tcq.h>
#include <scsi/scsi_transport_fc.h>

#include "lpfc_hw.h"
#include "lpfc_sli.h"
#include "lpfc_disc.h"
#include "lpfc_scsi.h"
#include "lpfc.h"
#include "lpfc_crtn.h"
#include "lpfc_logmsg.h"
#include "lpfc_ioctl.h"
#include "lpfc_diag.h"
#include "lpfc_misc.h"

extern struct list_head lpfcdfc_hosts;
extern struct mutex lpfcdfc_lock;

int
lpfc_issue_ct_rsp(struct lpfc_hba * phba, uint32_t tag,
				struct lpfc_dmabuf * bmp, DMABUFEXT_t * inp)
{
	struct lpfc_sli *psli;
	IOCB_t *icmd;
	struct lpfc_iocbq *ctiocb;
	struct lpfc_sli_ring *pring;
	uint32_t num_entry;
	unsigned long iflag;
	int rc = 0;

	spin_lock_irqsave(phba->host->host_lock, iflag);
	psli = &phba->sli;
	pring = &psli->ring[LPFC_ELS_RING];
	num_entry = inp->flag;
	inp->flag = 0;

	/* Allocate buffer for  command iocb */
	ctiocb = lpfc_sli_get_iocbq(phba);
	if (!ctiocb) {
		rc = ENOMEM;
		goto issue_ct_rsp_exit;
	}
	icmd = &ctiocb->iocb;

	icmd->un.xseq64.bdl.ulpIoTag32 = 0;
	icmd->un.xseq64.bdl.addrHigh = putPaddrHigh(bmp->phys);
	icmd->un.xseq64.bdl.addrLow = putPaddrLow(bmp->phys);
	icmd->un.xseq64.bdl.bdeFlags = BUFF_TYPE_BDL;
	icmd->un.xseq64.bdl.bdeSize = (num_entry * sizeof (struct ulp_bde64));
	icmd->un.xseq64.w5.hcsw.Fctl = (LS | LA);
	icmd->un.xseq64.w5.hcsw.Dfctl = 0;
	icmd->un.xseq64.w5.hcsw.Rctl = FC_SOL_CTL;
	icmd->un.xseq64.w5.hcsw.Type = FC_COMMON_TRANSPORT_ULP;

	pci_dma_sync_single_for_device(phba->pcidev, bmp->phys, LPFC_BPL_SIZE,
							PCI_DMA_TODEVICE);

	/* Fill in rest of iocb */
	icmd->ulpCommand = CMD_XMIT_SEQUENCE64_CX;
	icmd->ulpBdeCount = 1;
	icmd->ulpLe = 1;
	icmd->ulpClass = CLASS3;
	icmd->ulpContext = (ushort) tag;
	icmd->ulpTimeout = phba->fc_ratov * 2;

	/* Xmit CT response on exchange <xid> */
	lpfc_printf_log(phba,
			KERN_INFO,
			LOG_ELS,
			"%d:0118 Xmit CT response on exchange x%x Data: x%x "
			"x%x\n",
			phba->brd_no,
			icmd->ulpContext, icmd->ulpIoTag, phba->hba_state);

	ctiocb->iocb_cmpl = 0;
	ctiocb->iocb_flag |= LPFC_IO_LIBDFC;
	rc = lpfc_sli_issue_iocb_wait(phba, pring, ctiocb, 0,
				     phba->fc_ratov * 2 + LPFC_DRVR_TIMEOUT);

	if (rc == IOCB_TIMEDOUT) {
		ctiocb->context1 = NULL;
		ctiocb->context2 = NULL;
		ctiocb->iocb_cmpl = lpfc_ioctl_timeout_iocb_cmpl;
		spin_unlock_irqrestore(phba->host->host_lock, iflag);
		return (rc);
	}

	/* Calling routine takes care of IOCB_ERROR => EIO translation */
	if (rc != IOCB_SUCCESS)
		rc = IOCB_ERROR;

	lpfc_sli_release_iocbq(phba, ctiocb);
issue_ct_rsp_exit:
	spin_unlock_irqrestore(phba->host->host_lock, iflag);
	return (rc);
}

void
lpfc_sli_wake_mbox_wait(struct lpfc_hba * phba, LPFC_MBOXQ_t * pmboxq)
{
	wait_queue_head_t *pdone_q;

	/* 
	 * If pdone_q is empty, the driver thread gave up waiting and
	 * continued running.
	 */
	pdone_q = (wait_queue_head_t *) pmboxq->context1;
	if (pdone_q)
		wake_up_interruptible(pdone_q);
	return;
}

int
lpfc_sleep(struct lpfc_hba * phba, void *wait_q_head, long tmo)
{
	wait_queue_t wq_entry;
	int rc = 1;
	long left;

	init_waitqueue_entry(&wq_entry, current);
	/* start to sleep before we wait, to avoid races */
	set_current_state(TASK_INTERRUPTIBLE);
	add_wait_queue((wait_queue_head_t *) wait_q_head, &wq_entry);
	if (tmo > 0) {
		left = schedule_timeout(tmo * HZ);
	} else {
		schedule();
		left = 0;
	}
	remove_wait_queue((wait_queue_head_t *) wait_q_head, &wq_entry);

	if (signal_pending(current))
		return (EINTR);
	if (rc > 0)
		return (0);
	else
		return (ETIMEDOUT);
}

int
lpfc_geportname(struct lpfc_name * pn1, struct lpfc_name * pn2)
{
	int i;
	uint8_t *cp1, *cp2;

	i = sizeof (struct lpfc_name);
	cp1 = (uint8_t *) pn1;
	cp2 = (uint8_t *) pn2;
	while (i--) {
		if (*cp1 < *cp2) {
			return (0);
		}
		if (*cp1 > *cp2) {
			return (1);
		}
		cp1++;
		cp2++;
	}

	return (2);		/* equal */
}

/* Search for a nodelist entry on a specific list */
struct lpfc_nodelist *
lpfc_findnode_wwnn(struct lpfc_hba * phba, uint32_t order,
		   struct lpfc_name * wwnn)
{
	struct lpfc_nodelist *ndlp;
	struct list_head * lists[]={&phba->fc_nlpunmap_list,
				    &phba->fc_nlpmap_list};
	uint32_t search[]={NLP_SEARCH_UNMAPPED, NLP_SEARCH_MAPPED};
	uint32_t data1;
	int i;

	spin_lock_irq(phba->host->host_lock);
	for (i = 0; i < ARRAY_SIZE(lists); i++ ) {
		if (!(order & search[i]))
			continue;
		list_for_each_entry(ndlp, lists[i], nlp_listp)
			if (memcmp(&ndlp->nlp_nodename, wwnn,
				   sizeof(struct lpfc_name)) == 0) {
				spin_unlock_irq(phba->host->host_lock);
				data1 = (((uint32_t) ndlp->nlp_state << 24) |
					 ((uint32_t) ndlp->nlp_xri << 16) |
					 ((uint32_t) ndlp->nlp_type << 8) |
					 ((uint32_t) ndlp->nlp_rpi & 0xff));
				/* FIND node DID unmapped */
				lpfc_printf_log(phba,
						KERN_INFO,
						LOG_NODE,
						"%d:0911 FIND node by WWNN"
						" Data: x%p x%x x%x x%x\n",
						phba->brd_no,
						ndlp, ndlp->nlp_DID,
						ndlp->nlp_flag, data1);
				return ndlp;
			}
	}
	spin_unlock_irq(phba->host->host_lock);
	return NULL;
}


static uint32_t
lpfcdfc_host_event_next_put (struct lpfcdfc_host * dfchba)
{
	uint32_t put;

	mutex_lock (&lpfcdfc_lock);

	put = dfchba->hbaevt.put;

	if (++(dfchba->hbaevt.put)  >= LPFC_MAX_EVENT)
		dfchba->hbaevt.put = 0;

	if (dfchba->hbaevt.put == dfchba->hbaevt.get &&
	    ++(dfchba->hbaevt.get) >= LPFC_MAX_EVENT) {
		dfchba->hbaevt.get = 0;
		dfchba->hbaevt.missed++;
	}

	dfchba->hbaevt.cnt++;

	mutex_unlock (&lpfcdfc_lock);

	return put;
}

static void
lpfcdfc_host_event_put(struct lpfcdfc_host * dfchba,
		       enum lpfc_host_event_code evt,
		       uint32_t data)
{
	uint32_t put_index;
	
	put_index = lpfcdfc_host_event_next_put(dfchba);
	dfchba->hbaevt.buf[put_index].seq_num    = dfchba->hbaevt.cnt;
	dfchba->hbaevt.buf[put_index].event_code = evt;
	dfchba->hbaevt.buf[put_index].data       = data;
	return;
}

static inline void lpfcdfc_dfchba_event(struct lpfcdfc_host * dfchba,
					enum lpfc_host_event_code evt_code,
					uint32_t data)
{
	uint32_t evt_type;
	uint32_t evt_req_id = 0;
	struct lpfcdfc_event * evt;
	struct event_data * evt_dat = NULL;

	/* Store event for polling API */
	lpfcdfc_host_event_put(dfchba, evt_code, data);

	/* TODO:  wakeup waiting threads here */
	switch (evt_code) {
	case LPFCH_EVT_VENDOR_UNIQUE:
		evt_type = FC_REG_DUMP_EVENT;
		break;
	case LPFCH_EVT_LINKUP:
	case LPFCH_EVT_LINKDOWN:
		evt_type = FC_REG_LINK_EVENT;
		break;
	case LPFCH_EVT_RSCN:
		evt_type = FC_REG_RSCN_EVENT;
		break;
	case LPFCH_EVT_PORT_ONLINE:
	case LPFCH_EVT_PORT_OFFLINE:
	default:
		return;
	}
	mutex_lock (&lpfcdfc_lock);
	list_for_each_entry(evt, &dfchba->ev_waiters, node) {
		if (!(evt->type_mask & evt_type) ||
		    evt->req_id != evt_req_id)
			continue;

		lpfcdfc_event_ref(evt);

		if ((evt_dat = kzalloc(sizeof(*evt_dat), GFP_KERNEL)) == NULL) {
			lpfcdfc_event_unref(evt);
			break;
		}

		if (evt_type == FC_REG_LINK_EVENT &&
		    (evt_dat->data = kzalloc(sizeof(LinkInfo),
					     GFP_KERNEL)) != NULL) {
			lpfc_linkinfo(dfchba->phba,
				      (LinkInfo *)evt_dat->data);
			evt_dat->len = sizeof(LinkInfo);
		}

		evt_dat->immed_dat = data;
		evt_dat->type = evt_type;
		list_add(&evt_dat->node, &evt->events_to_see);
		wake_up_interruptible(&evt->wq);
		lpfcdfc_event_unref(evt);
	}
	mutex_unlock (&lpfcdfc_lock);

	return;
}

#ifdef HBA_EVENTS_SUPPORTED

void lpfcdfc_host_event(struct lpfc_hba * phba, enum lpfc_host_event_code evt,
                        uint32_t data)
{
	struct lpfcdfc_host * dfchba = lpfcdfc_host_from_hba(phba);

	BUG_ON(&dfchba->node == &lpfcdfc_hosts);

	lpfcdfc_dfchba_event(dfchba, evt, data);

	return;
}
#else
static int lpfcdfc_check_events(struct lpfcdfc_host * dfchba)
{
	struct Scsi_Host * host;
	int signaled_events = 0;

	if (dfchba->host == NULL)
		return signaled_events;

	if (dfchba->dev->driver && try_module_get(dfchba->dev->driver->driver.owner)) {
		host = pci_get_drvdata(dfchba->dev);
		if (host == NULL || (struct lpfc_hba*)host->hostdata != dfchba->phba) {
			dfchba->phba = NULL;
			dfchba->host = NULL;
			module_put(dfchba->dev->driver->driver.owner);
			/* TODO: signal hot unplug */
			return signaled_events;
		}
	} else
		return signaled_events;
	
	if ((dfchba->last_hba_state != LPFC_HBA_ERROR
	     && dfchba->phba->hba_state == LPFC_HBA_ERROR)
	    && !(dfchba->kill_active)) {
		lpfcdfc_dfchba_event(dfchba, LPFCH_EVT_VENDOR_UNIQUE, 0);
		/* update last_xxx vars -- no more signals to signal */
		// dfchba->last_hba_state = phba->hba_state;
		// dfchba->last_fc_flag = phba->fc_flag;
		signaled_events = 1;
	}		

	if (dfchba->last_fc_flag & FC_OFFLINE_MODE
	    && !(dfchba->phba->fc_flag & FC_OFFLINE_MODE)) {
		signaled_events++;
		lpfcdfc_dfchba_event(dfchba, LPFCH_EVT_PORT_ONLINE, 0);
	} else if (!(dfchba->last_fc_flag & FC_OFFLINE_MODE)
		   && dfchba->phba->fc_flag & FC_OFFLINE_MODE) {
		signaled_events++;
		lpfcdfc_dfchba_event(dfchba, LPFCH_EVT_PORT_OFFLINE, 0);
	}

	if (dfchba->last_hba_state < LPFC_LINK_UP
	    && dfchba->phba->hba_state >= LPFC_LINK_UP) {
		signaled_events++;
		lpfcdfc_dfchba_event(dfchba, LPFCH_EVT_LINKUP, 0);
	} else if (dfchba->last_hba_state >= LPFC_LINK_UP
	    && dfchba->phba->hba_state < LPFC_LINK_UP) {
		signaled_events++;
		lpfcdfc_dfchba_event(dfchba, LPFCH_EVT_LINKDOWN, 0);
	}

	dfchba->last_hba_state = dfchba->phba->hba_state;
	dfchba->last_fc_flag = dfchba->phba->fc_flag;

	module_put(dfchba->dev->driver->driver.owner);

	return signaled_events;
}

int lpfcdfc_do_ev_poll(void * p)
{
	struct list_head * list_head = p;
	struct lpfcdfc_host * dfchba;
	struct lpfcdfc_host * next_dfchba;

	DECLARE_WAIT_QUEUE_HEAD(wqh);
	DEFINE_WAIT(wq);

	set_user_nice(current, -20);
	// phba->work_wait = &wqh;

	for (;;) {
		prepare_to_wait(&wqh, &wq, TASK_UNINTERRUPTIBLE);

		if (kthread_should_stop())
			break;

		mutex_lock (&lpfcdfc_lock);
		list_for_each_entry(dfchba, list_head, node) {
			mutex_unlock (&lpfcdfc_lock);
			lpfcdfc_check_events(dfchba);
			mutex_lock (&lpfcdfc_lock);
		}
		list_for_each_entry_safe(dfchba, next_dfchba, list_head, node) {
			if (dfchba->host == NULL) {
				list_del_init(&dfchba->node);
				pci_dev_put(dfchba->dev);
				kfree (dfchba);
			}
		}
		mutex_unlock (&lpfcdfc_lock);

		schedule_timeout(HZ/2);

		if (signal_pending(current))
			break;
	}
	finish_wait(&wqh, &wq);
	return 0;
}
#endif

void lpfcdfc_els_unsol_event(struct lpfc_hba * phba, struct lpfc_sli_ring * pring,
			     struct lpfc_iocbq * piocbq)
{
	struct lpfcdfc_host * dfchba = lpfcdfc_host_from_hba(phba);
	struct lpfc_dmabuf *dmabuf = NULL;
	dma_addr_t dma_addr;
	uint32_t len;
	uint32_t* cmd;
	int i;

	BUG_ON(&dfchba->node == &lpfcdfc_hosts);

	if (piocbq->iocb.ulpBdeCount > 0
	    && piocbq->iocb.un.cont64[0].tus.f.bdeSize > 0)
	{
		dma_addr = getPaddr(piocbq->iocb.un.cont64[0].addrHigh,
				    piocbq->iocb.un.cont64[0].addrLow);
		dmabuf = lpfc_sli_ringpostbuf_get(phba, pring, dma_addr);

		if (dmabuf == NULL)
			goto error_unsol_els_exit;

		cmd = (uint32_t *)(dmabuf->virt);

		if ((*cmd & ELS_CMD_MASK) != ELS_CMD_RSCN) {
			lpfc_sli_ringpostbuf_put(phba, pring, dmabuf);
			goto error_unsol_els_exit;
		}

		len =  be32_to_cpu(*cmd) & 0xffff;
		len -= sizeof (uint32_t);
		cmd++;

		for (i = 0; i < len/sizeof(uint32_t); i++)
			lpfcdfc_dfchba_event(dfchba, LPFCH_EVT_RSCN, cmd[i]);

		lpfc_sli_ringpostbuf_put(phba, pring, dmabuf);
	}

error_unsol_els_exit:
	/* TODO:  wake_up waiting threads here */
	if (dfchba->base_els_unsol_event != NULL)
		(dfchba->base_els_unsol_event)(phba, pring, piocbq);

	return;
}

void lpfcdfc_ct_unsol_event(struct lpfc_hba * phba,
			    struct lpfc_sli_ring * pring,
			    struct lpfc_iocbq * piocbq)
{
	struct lpfcdfc_host * dfchba = lpfcdfc_host_from_hba(phba);
	uint32_t evt_type;
	uint32_t evt_req_id = 0;
	uint32_t cmd;
	uint32_t len;
	struct lpfc_dmabuf *dmabuf = NULL;
	struct lpfcdfc_event * evt;
	struct event_data * evt_dat = NULL;
	struct lpfc_iocbq * iocbq;
	size_t offset = 0;
	struct list_head head;
	struct ulp_bde64 * bde;
	dma_addr_t dma_addr;
	int i;

	BUG_ON(&dfchba->node == &lpfcdfc_hosts);
	INIT_LIST_HEAD(&head);
	evt_type = FC_REG_CT_EVENT;
	if (piocbq->iocb.ulpBdeCount > 0
	    && piocbq->iocb.un.cont64[0].tus.f.bdeSize > 0)
	{
		dma_addr = getPaddr(piocbq->iocb.un.cont64[0].addrHigh,
				    piocbq->iocb.un.cont64[0].addrLow);
		dmabuf = lpfc_sli_ringpostbuf_get(phba, pring, dma_addr);
		BUG_ON(dmabuf == NULL);
		evt_req_id =
			((struct lpfc_sli_ct_request *)(dmabuf->virt))->FsType;
		cmd = ((struct lpfc_sli_ct_request *)(dmabuf->virt))->CommandResponse.bits.CmdRsp;
		len = ((struct lpfc_sli_ct_request *)(dmabuf->virt))->CommandResponse.bits.Size;
		lpfc_sli_ringpostbuf_put(phba, pring, dmabuf);
	} else
		goto error_unsol_ct_exit;

	mutex_lock (&lpfcdfc_lock);
	list_for_each_entry(evt, &dfchba->ev_waiters, node) {
		if (!(evt->type_mask & FC_REG_CT_EVENT) ||
		    evt->req_id != evt_req_id)
			continue;

		lpfcdfc_event_ref(evt);

		if ((evt_dat = kzalloc(sizeof(*evt_dat), GFP_KERNEL)) == NULL) {
			lpfcdfc_event_unref(evt);
			break;
		}

		INIT_LIST_HEAD(&head);
		list_add_tail(&head, &piocbq->list);
		iocbq = piocbq;
		list_for_each_entry(iocbq, &head, list)
			for (i = 0; i < iocbq->iocb.ulpBdeCount; i++)
				evt_dat->len +=
					iocbq->iocb.un.cont64[i].tus.f.bdeSize;

		evt_dat->data = kzalloc(evt_dat->len, GFP_KERNEL);
		if (evt_dat->data == NULL) {
			kfree (evt_dat);
			lpfcdfc_event_unref(evt);
			mutex_unlock (&lpfcdfc_lock);
			goto error_unsol_ct_exit;
		}

		iocbq = piocbq;
		list_for_each_entry(iocbq, &head, list)
			for (i = 0; i < iocbq->iocb.ulpBdeCount; i++) {
				bde = &iocbq->iocb.un.cont64[i];
				dma_addr = getPaddr(bde->addrHigh,
						    bde->addrLow);
				dmabuf = lpfc_sli_ringpostbuf_get(phba, pring,
								  dma_addr);
				if (dmabuf == NULL) {
					kfree (evt_dat->data);
					kfree (evt_dat);
					lpfcdfc_event_unref(evt);
					mutex_unlock (&lpfcdfc_lock);
					goto error_unsol_ct_exit;
				}
				memcpy ((char *)(evt_dat->data) + offset,
					dmabuf->virt, bde->tus.f.bdeSize);
				offset += bde->tus.f.bdeSize;
				if (evt_req_id != SLI_CT_ELX_LOOPBACK)
					lpfc_sli_ringpostbuf_put(phba, pring,
								 dmabuf);
				else {
					switch (cmd) {
					case ELX_LOOPBACK_DATA:
						dfc_cmd_data_free(phba, (DMABUFEXT_t *)dmabuf);
						break;
					case ELX_LOOPBACK_XRI_SETUP:
					default:
						lpfc_post_buffer(phba, pring, 1, 1);
						lpfc_mbuf_free(phba, dmabuf->virt,
							       dmabuf->phys);
						kfree(dmabuf);
						break;
					};
				}
			}

		evt_dat->immed_dat = piocbq->iocb.ulpContext;
		evt_dat->type = evt_type;
		list_add(&evt_dat->node, &evt->events_to_see);
		wake_up_interruptible(&evt->wq);
		lpfcdfc_event_unref(evt);
		if (evt_req_id == SLI_CT_ELX_LOOPBACK)
			break;
	}
	mutex_unlock (&lpfcdfc_lock);

error_unsol_ct_exit:
	if(!list_empty(&head))
		list_del(&head);
	if (evt_req_id != SLI_CT_ELX_LOOPBACK &&
	    dfchba->base_ct_unsol_event != NULL)
		(dfchba->base_ct_unsol_event)(phba, pring, piocbq);

	return;
}


DMABUFEXT_t *
dfc_cmd_data_alloc(struct lpfc_hba * phba,
		   char *indataptr, struct ulp_bde64 * bpl, uint32_t size)
{
	DMABUFEXT_t *mlist = 0;
	DMABUFEXT_t *dmp;
	int cnt, offset = 0, i = 0;
	struct pci_dev *pcidev;

	pcidev = phba->pcidev;

	while (size) {
		/* We get chucks of 4K */
		if (size > 4096)
			cnt = 4096;
		else
			cnt = size;

		/* allocate DMABUFEXT_t buffer header */
		dmp = kmalloc(sizeof (DMABUFEXT_t), GFP_KERNEL);
		if ( dmp == 0 ) {
			goto out;
		}

		INIT_LIST_HEAD(&dmp->dma.list);

		/* Queue it to a linked list */
		if (mlist)
			list_add_tail(&dmp->dma.list, &mlist->dma.list);
		else
			mlist = dmp;

		/* allocate buffer */
		dmp->dma.virt = dma_alloc_coherent(&pcidev->dev, 
						   cnt, 
						   &(dmp->dma.phys), 
						   GFP_KERNEL);

		if (dmp->dma.virt == 0) {
			goto out;
		}
		dmp->size = cnt;

		if (indataptr) {
			/* Copy data from user space in */
			if (copy_from_user
			    ((uint8_t *) dmp->dma.virt,
			     (uint8_t *) (indataptr + offset), (ulong) cnt)) {
				goto out;
			}
			bpl->tus.f.bdeFlags = 0;

			pci_dma_sync_single_for_device(phba->pcidev,
			        dmp->dma.phys, LPFC_BPL_SIZE, PCI_DMA_TODEVICE);

		} else {
			memset((uint8_t *)dmp->dma.virt, 0, cnt);
			bpl->tus.f.bdeFlags = BUFF_USE_RCV;
		}

		/* build buffer ptr list for IOCB */
		bpl->addrLow = le32_to_cpu( putPaddrLow(dmp->dma.phys) );
		bpl->addrHigh = le32_to_cpu( putPaddrHigh(dmp->dma.phys) );
		bpl->tus.f.bdeSize = (ushort) cnt;
		bpl->tus.w = le32_to_cpu(bpl->tus.w);
		bpl++;

		i++;
		offset += cnt;
		size -= cnt;
	}

	mlist->flag = i;
	return (mlist);
out:
	dfc_cmd_data_free(phba, mlist);
	return (0);
}

int
dfc_cmd_data_free(struct lpfc_hba * phba, DMABUFEXT_t * mlist)
{
	DMABUFEXT_t *mlast;
	struct pci_dev *pcidev;
	struct list_head head, *curr, *next;

	if (!mlist)
		return(0);

	pcidev = phba->pcidev;
	list_add_tail(&head, &mlist->dma.list);

	list_for_each_safe(curr, next, &head) {
		mlast = list_entry(curr, DMABUFEXT_t , dma.list);
		if (mlast->dma.virt) {

			dma_free_coherent(&pcidev->dev, 
					  mlast->size, 
					  mlast->dma.virt, 
					  mlast->dma.phys);

		}
		kfree(mlast);
	}
	return (0);
}
